/*
  Modified AR-40 Rotator Controller
  written by Glen Popiel - KW5GP

*/

// #define debug  // Enables Debug Mode - Sends Debug output to Serial Monitor
// #define debug1 // Sends adc debug data to Serial Monitor

#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library
#include <SPI.h>
#include <EEPROM.h>  // Include EEPROM Library
#include <ADS1115.h>

const int comm_speed = 19200;  // Set the Serial Monitor Baud Rate

#define Sense_24VAC 2  // Assign 24VAC Sense input to Pin 2
#define left 3   // Assign Left (Counter Clockwise) Relay to Pin 3
#define right 4  // Assign Right (Clockwise) Relay to Pin 4
#define cal_zero 9    // Assign Zero Calibrate switch to Pin 9
#define cal_360 5     // Assign 360 Calibrate Switch to Pin 5
#define pos_sense A0 // Assign the position sensor input pin to A0
#define TFT_CS     6  // Assign the TFT CS to Pin 6
#define TFT_RST    7  // Assign the TFT RST to Pin 7
#define TFT_DC     8  // Assign the TFT DC to Pin 8
#define pos_sense_ctl 10  // Assign the Position Sense Control to pin 10

#define EEPROM_ID_BYTE 1   // EEPROM ID to validate EEPROM data location
#define EEPROM_ID  54  // EEPROM ID Value
#define EEPROM_AZ_CAL_0 2    // Azimuth Zero Calibration EEPROM location     
#define EEPROM_AZ_CAL_MAX 4  // Azimuth Max Calibration Data EEPROM location  

#define AZ_CAL_0_DEFAULT 30 // Define the default Cal 0 value
#define AZ_CAL_MAX_DEFAULT 25000 // Define the default Cal Max value

#define AZ_Tolerance 1  // Set the Azimuth Accuracy Tolerance

int long current_AZ;  // Variable for current Azimuth ADC Value
int AZ_Degrees;  // Variable for Current Azimuth in Degrees
boolean moving = false;  // Variable to let us know if the rotor is moving
int Arduino_mode = -1; // Variable to let us know the Controller Switch is in Arduino Mode
int Previous_mode = -2; // Variable to track mode switch
int long previous_AZ = -1;  // Variable to track the previous AZ reading
String Azimuth = "   ";
boolean calibrate = false;
boolean first_pass = true;  // Variable used to determine if first pass through the main loop

// The Azimuth Correction arrays
const int  AZ_Correction[12] = { -550, -1500, -2100, -1900, -2350, -2400, -2800, -2500, -2000 , -1850, -1000, 27};

const int  R_correct[12] = { -700, -1600, -2300, -2100, -2500, -2600, -3000, -2700, -2200 , -2000, -1300, -50};

const int  L_correct[12] = { -550, -1350, -1900, -1700, -2200, -2200, -2600, -2300, -1800 , -1730, -700, -900};

const int r_zero_adj = 0;
const int l_zero_adj = -210;
const int r_max_adj = 0;
const int l_max_adj = 0;

const int AZ_Max_Correction = 30;

int set_AZ;  // Azimuth set value
int AZ_0;  // Azimuth Zero Value from EEPROM
int AZ_MAX; // Azimuth Max Value from EEPROMs
int zone_size; // Approximate number of A/D counts per zone
int zone = 0; // Current rotation zone

int long update_time = 0;
const int update_delay = 1000;
boolean turn_signal = false;
String previous_direction = "S";

String rotate_direction;

byte inByte = 0;  // incoming serial byte
byte serial_buffer[50];  // incoming serial byte buffer
int serial_buffer_index = 0;  // The index pointer variable for the Serial buffer
String Serial_Send_Data; // Data to send to Serial Port
String Requested_AZ; // RS232 Requested Azimuth - M and short W command
int AZ_To; // Requested AZ Move
int AZ_Distance; // Distance to move AZ

ADS1115 adc;  // Define the A/D as adc

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS,  TFT_DC, TFT_RST);  // Define the ST7735 TFT as tft

void setup()
{
  // initialize the digital I/O pin modes
  pinMode(right, OUTPUT);
  pinMode(left, OUTPUT);
  pinMode(pos_sense_ctl, OUTPUT);
  pinMode(Sense_24VAC, INPUT_PULLUP);
  pinMode(cal_zero, INPUT_PULLUP);
  pinMode(cal_360, INPUT_PULLUP);

  digitalWrite(pos_sense_ctl, LOW); // Turn off the position sensor control relay
  digitalWrite(right, LOW); // Turn off the rotator control relays
  digitalWrite(left, LOW);

  Wire.begin(); // Join the I2C Bus

  adc.initialize(); // Initialize the ADS1115 16 bit A/D module
  Wire.beginTransmission(0x48); // Begin direct communication with ADC
  Wire.write(0x1);  // Connect to the ADC and send two bytes - Set the config register to all 1's
  Wire.write(0x7F); // MSB
  Wire.write(0xFF); // LSB
  Wire.endTransmission(); // End the direct ADC comms

  adc.setMode(ADS1115_MODE_CONTINUOUS); // free running conversion
  adc.setGain(ADS1115_PGA_2P048);  // set adc gain to 2.048v range, .0625 mv/step
  adc.setRate(ADS1115_RATE_128);  // set adc sample rate to 128 samples per second
  adc.setMultiplexer(ADS1115_MUX_P0_NG);  // AN0+ Vs ground - Single mode input on A0

  // Use this initializer if you're using a 1.8" TFT
  tft.initR(INITR_BLACKTAB);   // initialize a ST7735S chip, black tab

  // Use this initializer (uncomment) if you're using a 1.44" TFT
  //tft.initR(INITR_144GREENTAB);   // initialize a ST7735S chip, green tab

  Serial.begin(comm_speed);  // Start the Serial port

  tft.fillScreen(ST7735_BLUE); // Clear the display
  tft.setRotation(1); // Set the screen rotation
  tft.setTextWrap(false);
  tft.setTextSize(3);
  tft.setTextColor(ST7735_GREEN);
  tft.setCursor(40, 10);
  tft.print("KW5GP");
  tft.setTextSize(2);
  tft.setCursor(55, 60);
  tft.print("AR-40");
  tft.setCursor(45, 80);
  tft.print("Rotator" );
  tft.setCursor(25, 100);
  tft.print("Controller");

  read_eeprom_cal_data();  // Read the EEPROM calibration data

#ifdef debug  // Display the calibration data when in debug mode
  Serial.print("ROM Cal - Zero: ");
  Serial.print(AZ_0);
  Serial.print(" Max: ");
  Serial.print(AZ_MAX);
#endif

  zone_size = AZ_MAX / 12;
  AZ_MAX = AZ_MAX - AZ_Max_Correction;

#ifdef debug
  Serial.print(" Adj Max: ");
  Serial.print(AZ_MAX);
  Serial.print(" Zone Size: ");
  Serial.println(zone_size);
#endif

  delay(5000);  //Wait 5 seconds then clear the startup message
  tft.fillScreen(ST7735_BLUE); // Clear the display

  set_AZ = -1;  // Preset the Azimuth Move Variables

  Arduino_mode = !(digitalRead(Sense_24VAC));
  if (Arduino_mode == 1)
  {
    // We started in Arduino mode
    // Update the display - which also turns on the position sensor control relay and clears the A/D
    Previous_mode = Arduino_mode;
    digitalWrite(pos_sense_ctl, HIGH);
    delay(1000);

#ifdef debug
    Serial.println();
    Serial.println("Start in Arduino Mode");
#endif

    update_display();
    first_pass = false;
  }
}  // End Setup Loop

void loop()
{
  Arduino_mode = !(digitalRead(Sense_24VAC)); // Check to see if we're in manual or Arduino mode

  if (Arduino_mode != Previous_mode)
  {

#ifdef debug
    Serial.print("24VAC Sense: ");
    Serial.println(Arduino_mode);
#endif

    if (Arduino_mode == 1)
    {
      // We just switched into Arduino mode - update the display
      update_display();
    }
  }

  if (Arduino_mode == 1)
  {
    //    read_adc();
    update_display();
    check_cal();  // Check to see if either calibration switch is pressed
    check_serial();
    check_move();
  } else
  {
    offline_display();
  }
  Previous_mode = Arduino_mode;

#ifdef debug1
  delay(250);
#endif

}  // End Main Loop

void check_cal()  // Checks to see if a calibration switch is pressed and set calibration accordingly
{
  if (digitalRead(cal_zero) == 0) // Cal Zero button pressed
  {
    set_0_az_cal();
  }
  if (digitalRead(cal_360) == 0) // Cal Zero button pressed
  {
    set_max_az_cal();
  }
}

void all_stop() // Relay Off function - stops motion, delays 3 seconds then turns off Brake message
{
  tft.setTextSize(2);
  clear_top_line();
  tft.setCursor(40, 3); // Display message on LCD
  tft.print("Braking");
  digitalWrite(right, LOW);  // Turn off CW Relay
  digitalWrite(left, LOW);  // Turn off CCW Relay
  moving = false;
  rotate_direction = "S";  // Set direction to S (Stop)
  delay(3000);

  clear_top_line();
}

void read_eeprom_cal_data()  // Read the EEPROM Calibration data
{
  if (EEPROM.read(EEPROM_ID_BYTE) == EEPROM_ID)   // Verify the EEPROM has valid data
  {

#ifdef debug // Display the Calibration data in debug mode
    Serial.print("Read EEPROM Cal Data Valid - AZ_CAL_0: ");
    Serial.print((EEPROM.read(EEPROM_AZ_CAL_0) * 256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1), DEC);
    Serial.print("   AZ_CAL_MAX: ");
    Serial.println((EEPROM.read(EEPROM_AZ_CAL_MAX) * 256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1), DEC);
#endif

    AZ_0 = (EEPROM.read(EEPROM_AZ_CAL_0) * 256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1); // Set the Zero degree Calibration Point
    AZ_MAX = (EEPROM.read(EEPROM_AZ_CAL_MAX) * 256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1); // Set the 360 degree Calibration Point

  } else
  { // EEPROM has no Calibration data - initialize eeprom to default values

#ifdef debug
    // Send status message in debug mode
    Serial.println("Read EEPROM Cal Data Invalid - set to defaults");
#endif

    AZ_0 = AZ_CAL_0_DEFAULT;  // Set the Calibration data to default values
    AZ_MAX = AZ_CAL_MAX_DEFAULT;
    write_eeprom_cal_data();  // Write the data to the EEPROM
  }
}

void write_eeprom_cal_data() // Write the Calibration data to the EEPROM
{
#ifdef debug
  Serial.println("Writing EEPROM Cal Data");  // Display status in debug mode
#endif

  EEPROM.write(EEPROM_ID_BYTE, EEPROM_ID); // Write the EEPROM ID to the EEPROM
  EEPROM.write(EEPROM_AZ_CAL_0, highByte(AZ_0)); // Write Zero Calibration Data High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_0 + 1, lowByte(AZ_0)); // Write Zero Calibration Data Low Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX, highByte(AZ_MAX)); // Write 360 Calibration Data High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX + 1, lowByte(AZ_MAX)); // Write 360 Calibration Data Low Order Byte
}

void check_serial() // Function to check for data on the Serial port
{
  if (Serial.available() > 0) // Get the Serial Data if available
  {
    inByte = Serial.read();  // Get the Serial Data

    // You may need to uncomment the following line if your PC software
    // will not communicate properly with the controller
    
 //   Serial.print(char(inByte));  // Echo back to the PC

    if (inByte == 10)  // ignore Line Feeds
    {
      return;
    }
    if (inByte != 13) // Add to buffer if not CR
    {
      serial_buffer[serial_buffer_index] = inByte;

#ifdef debug // Print the Character received if in Debug mode
      Serial.print("RX: ");
      Serial.print(" *** ");
      Serial.print(serial_buffer[serial_buffer_index]);
      Serial.println(" *** ");
#endif

      serial_buffer_index++;  // Increment the Serial Buffer pointer

    } else
    { // It's a Carriage Return, execute command

      if ((serial_buffer[0] > 96) && (serial_buffer[0] < 123))  //If first character of command is lowercase, convert to uppercase
      {
        serial_buffer[0] = serial_buffer[0] - 32;
      }

      switch (serial_buffer[0]) {  // Decode first character of command

        case 65:  // A Command - Stop the Azimuth Rotation

#ifdef debug
          Serial.println("A RX");
#endif

          az_rotate_stop();
          break;

        case 67:      // C - return current azimuth

#ifdef debug
          Serial.println("C RX ");
#endif

          send_current_az();  // Return Azimuth if C Command
          break;

        case 70:  // F - Set the Max Calibration

#ifdef debug
          Serial.println("F RX");
          Serial.println(serial_buffer_index);
#endif

          set_max_az_cal();  // F - Set the Max Azimuth Calibration
          break;

        case 76:  // L - Rotate Azimuth CCW

#ifdef debug
          Serial.println("L RX");
#endif

          rotate_az_ccw();  // Call the Rotate Azimuth CCW Function
          break;

        case 77:  // M - Rotate to Set Point

#ifdef debug
          Serial.println("M RX");
#endif

          rotate_to();  // Call the Rotate to Set Point Function
          break;

        case 79:  // O - Set Zero Calibration

#ifdef debug
          Serial.println("O RX");
          Serial.println(serial_buffer_index);
#endif

          set_0_az_cal();  // O - Set the Azimuth Zero Calibration
          break;

        case 82:  // R - Rotate Azimuth CW

#ifdef debug
          Serial.println("R RX");
#endif

          rotate_az_cw();  // Call the Rotate Azimuth CW Function
          break;

        case 83:  // S - Stop All Rotation

#ifdef debug
          Serial.println("S RX");
#endif

          az_rotate_stop();  // Call the Stop Azimith Rotation Function
          break;

      }

      serial_buffer_index = 0;  // Clear the Serial Buffer and Reset the Buffer Index Pointer
      serial_buffer[0] = 0;
    }
  }
}

void check_move() // Check to see if we've been commanded to move
{
  if (set_AZ != -1)
  { // We're moving - check and stop as needed

    read_adc();

    // Map AZ to degrees

#ifdef debug
    Serial.print(" *** check_move *** AZ_To: ");
    Serial.print(AZ_To);
    Serial.print("  Zone: ");
    Serial.print(zone);
    Serial.print("  Zone Adj: ");
    if (rotate_direction == "R")
    {
      Serial.print(R_correct[zone]);
    } else
    {
      Serial.print(L_correct[zone]);
    }
    Serial.print(" Current_AZ: ");
    Serial.print(current_AZ);
    Serial.print(" AZ_Deg: ");
    Serial.print(AZ_Degrees);
#endif

    if (set_AZ != -1) // If Azimuth is moving
    {
      AZ_Distance = set_AZ - AZ_Degrees;  // Check how far we have to move

#ifdef debug
      Serial.print("  AZ_Distance: ");
      Serial.print(AZ_Distance);
#endif

      fix_180();

#ifdef debug
      Serial.print("  Adj AZ_Distance: ");
      Serial.print(AZ_Distance);
      Serial.print("  Direction: ");
      Serial.println(rotate_direction);
#endif

      if (abs(AZ_Distance) <= AZ_Tolerance)  // No move needed if we're within the tolerance range
      {
        az_rotate_stop();  // Stop the Azimuth Rotation
        set_AZ = -1;  // Turn off the Azimuth Move Command
      } else
      { // Move Azimuth - figure out which way
        if (AZ_Distance > 0)   //We need to move CW
        {
          rotate_az_cw();  // Rotate CW if positive
        } else {
          rotate_az_ccw();  // Rotate CCW if negative
        }
      }
    }
  }
}

void send_current_az() // Send the Current Azimuth Function
{
  read_adc();  // Read the ADC

#ifdef debug
  Serial.println();
  Serial.print("Deg: ");
  Serial.print(AZ_Degrees);
  Serial.print("  Return Value: ");
#endif

  // Send it back via serial
  Serial_Send_Data = "";
  if (AZ_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = "0";
  }
  if (AZ_Degrees < 10)
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(AZ_Degrees);  // Send the Azimuth in Degrees
  Serial.println(Serial_Send_Data);  // Return value via RS-232 port
}

void set_max_az_cal() // Set the Max Azimuth Calibration Function
{
#ifdef debug
  Serial.println("Cal Max AZ Function");
#endif

  calibrate = true;
  read_adc();  // Read the ADC

  // save current az value to EEPROM - Zero Calibration

#ifdef debug
  Serial.println(current_AZ);
#endif

  AZ_MAX = current_AZ;  // Set the Azimuth Maximum Calibration to Current Azimuth Reading
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM

#ifdef debug
  Serial.println("Max Azimuth Cal Complete");
#endif

  calibrate = false;
}

void rotate_az_ccw() // Function to Rotate Azimuth CCW
{
  digitalWrite(left, HIGH);  // Set the Rotate Left Pin High
  digitalWrite(right, LOW);  // Make sure the Rotate Right Pin is Low
  rotate_direction = "L";  // set the direction flag to "L" (Left - CCW)
  if (!turn_signal || (previous_direction != rotate_direction))
  {
    tft.setTextSize(2);
    clear_top_line();
    tft.setCursor(3, 10); // display the left arrow on the LCD
    tft.print("<==");
    turn_signal = true;
    previous_direction = rotate_direction;
  }
  moving = true;
}

void rotate_az_cw() // Function to Rotate Azimuth CW
{
  digitalWrite(right, HIGH);    // Set the Rotate Right Pin High
  digitalWrite(left, LOW);    // Make sure the Rotate Left Pin Low
  rotate_direction = "R";  // set the direction flag to "R" (Right - CW)
  if (!turn_signal || (previous_direction != rotate_direction))
  {
    tft.setTextSize(2);
    clear_top_line();
    tft.setCursor(120, 10); // display the left arrow on the LCD
    tft.print("==>");
    turn_signal = true;
    previous_direction = rotate_direction;
  }
  moving = true;
}

void az_rotate_stop() // Function to Stop Azimuth Rotation
{

  digitalWrite(right, LOW);  // Turn off the Rotate Right Pin
  digitalWrite(left, LOW);   // Turn off the Rotate Left Pin

  tft.setTextSize(2);
  clear_top_line();
  tft.setCursor(40, 3); // Display Braking message on LCD
  tft.print("Braking");
  set_AZ = -1;
  turn_signal = false;
  delay(3000);

  clear_top_line();

  moving = false;
  rotate_direction = "S";  // Set direction to S (Stop)
  previous_direction = rotate_direction;
}

void rotate_to() // Function to Rotate to Set Point
{
#ifdef debug
  Serial.println("M Command -  rotate_to Function");
  Serial.print("  Chars RX: ");
  Serial.print(serial_buffer_index);
#endif

  // Decode Command - Format Mxxx - xxx = Degrees to Move to
  if (serial_buffer_index == 4)  // Verify the Command is the proper length
  {
#ifdef debug
    Serial.print("  Value [1] to [3]: ");
#endif

    Requested_AZ = (String(char(serial_buffer[1])) + String(char(serial_buffer[2])) + String(char(serial_buffer[3]))) ;  // Decode the Azimuth Value
    AZ_To = (Requested_AZ.toInt()); // AZ Degrees to Move to as integer


#ifdef debug
    Serial.println(Requested_AZ);
    Serial.print("AZ_To: ");
    Serial.print(AZ_To);
#endif

    read_adc();  // Read the ADC

#ifdef debug
    Serial.print("  Zone: ");
    Serial.print(zone);
    Serial.print("  Current Deg: ");
    Serial.print(AZ_Degrees);
#endif

    AZ_Distance = AZ_To - AZ_Degrees;  // Figure out far we have to move

#ifdef debug
    Serial.print("  AZ_Dist: ");
    Serial.print(AZ_Distance);
#endif

    fix_180();

    set_AZ = AZ_To;
    moving = true;

#ifdef debug
    Serial.print("  Adj AZ_Dist: ");
    Serial.print(AZ_Distance);
#endif

    if (abs(AZ_Distance) <= AZ_Tolerance)  // No move needed if we're within the Tolerance Range
    {
      az_rotate_stop();  // Stop the Azimuth Rotation
      set_AZ = -1;  // Turn off the Move Command
    } else
    { // Move Azimuth - figure out which way
      if (AZ_Distance > 0)   // We need to move CW
      {

#ifdef debug
        Serial.println("  Turn right ");
#endif

        rotate_az_cw();  // If the distance is positive, move CW
      } else
      {

#ifdef debug
        Serial.println("  Turn left ");
#endif
        rotate_az_ccw();  // Otherwise, move counterclockwise
      }
    }
  }
}

void set_0_az_cal() // Set Azimuth Zero Calibration
{

#ifdef debug
  Serial.println("Cal Zero Function");
#endif

  calibrate = true;
  read_adc();  // Read the ADC
  // save current Azimuth value to EEPROM - Zero Calibration

#ifdef debug
  Serial.println(current_AZ);
#endif

  AZ_0 = current_AZ;  // Set the Azimuth Zero Calibration to current position
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM

#ifdef debug
  Serial.println("Zero AZ Cal Complete");
#endif

  calibrate = false;
}

void fix_az_string()  // Convert azimuth to string and pad to 3 characters
{
  Azimuth = "00" + String(AZ_Degrees);
  Azimuth = Azimuth.substring(Azimuth.length() - 3);
}

void update_display()
{

  if ((Arduino_mode != Previous_mode) || first_pass)
  {
    // Clear the screen and display normal operation
    tft.fillScreen(ST7735_BLUE); // Clear the display
    tft.setTextSize(1);
    tft.setTextColor(ST7735_GREEN);
    tft.setTextSize(1);
    tft.setCursor(25, 115);
    tft.print("Arduino Mode Online");
    digitalWrite(pos_sense_ctl, HIGH);

#ifdef debug
    Serial.println();
    Serial.println("Switch to Arduino mode - delay for ADC");
#endif

    previous_AZ = -1;  // Reset the previous AZ before starting
    Previous_mode = Arduino_mode;
    first_pass = false;
    delay(1000);

  }
  if (abs(millis()) > abs(update_time + update_delay))
  {
    read_adc();
    if (AZ_Degrees != previous_AZ)
    {
      tft.setTextSize(6);
      tft.fillRect(30, 30, 130, 55, ST7735_BLUE);
      tft.setTextColor(ST7735_GREEN);
      tft.setCursor(30, 40);
      tft.print(Azimuth);
      tft.setCursor(140, 30);
      tft.setTextSize(3);
      tft.print("o");
      previous_AZ = AZ_Degrees;
    }
    update_time = millis();
  }


#ifdef debug1
  Serial.print("    ");
  Serial.print(" Deg: ");
  Serial.println(AZ_Degrees); // Send position to Serial Monitor in debug mode
#endif


}

void offline_display()
{
  if (Arduino_mode != Previous_mode)
  {
    tft.fillScreen(ST7735_BLUE); // Clear the display
    tft.setTextSize(3);
    tft.setTextColor(ST7735_RED);
    tft.setCursor(40, 25);
    tft.print("AR-40");
    tft.setTextSize(2);
    tft.setCursor(25, 55);
    tft.print("Controller");
    tft.setCursor(45, 75);
    tft.print("Not In" );
    tft.setCursor(10 , 95);
    tft.print("Arduino Mode");
    digitalWrite(pos_sense_ctl, LOW);
  }
}

// Read the A/D Converter
void read_adc()
{
  // Read ADC and display position
  if (Arduino_mode != Previous_mode)  // First ADC read, we need to allow settle time
  {
    digitalWrite(pos_sense_ctl, HIGH);
    delay(1000);
  }
  adc.setGain(ADS1115_PGA_2P048);  // set adc gain to 2.048v range, .0625 mv/step
  adc.setRate(ADS1115_RATE_128);  // set adc sample rate to 128 samples per second

  // Read adc
  current_AZ = adc.getDiff0();  // Read ADC channel 0


#ifdef debug1
  Serial.print("   Read ADC: "); // display ADC read status in debug mode
  Serial.print(current_AZ);  // Display ADC value in debug mode
#endif

  if (calibrate)
  {
    return;
  }

  if (current_AZ < 0)
  {
    current_AZ = 0;
  }

  if (current_AZ > AZ_MAX)
  {
    current_AZ = AZ_MAX;
  }

  zone_correct(); // Adjust A/D based on rotation zone

  map_az();

  fix_az_string();  // Convert azimuth to string then pad to 3 characters and save in Azimuth

#ifdef debug1
  Serial.print(" Moving: "); Serial.print(moving);
  Serial.print(" set_AZ: "); Serial.print(set_AZ);
#endif

}

void clear_top_line()
{
  tft.fillRect(0, 0, 160, 30, ST7735_BLUE);
}


void zone_correct() // Adjusts A/D value based on rotation zone
{
  int start_zone;
  int end_zone;
  int correction;
  int start_pos;  // Start of current zone
  int end_pos;  // End of current zone
  int current_pos;  // current position in zone
  int calculated_AZ;  // Used to adjust current AZ at zero point

  // Figure out what rotation zone we're in

  zone = current_AZ / zone_size;

  if (zone < 0)
  {
    zone = 0;
  }
  if (zone > 11)
  {
#ifdef debug1
    Serial.println(" Zone forced to 11");
#endif
    zone = 11;
  }

  start_pos = (zone * zone_size);
  end_pos = start_pos + zone_size;

#ifdef debug1
  Serial.print("  Zone: ");
  Serial.print(zone);
#endif

  if (zone == 0)
  {
    start_zone = 0;
    if (!moving)
    {
      start_pos = AZ_0;
      end_zone = AZ_Correction[0];
    } else
    {
      // we're moving - use moving correction factors
      if (rotate_direction == "R")
      {
        // We're turning right
        start_pos = AZ_0;
        end_zone = R_correct[0];
        current_AZ = current_AZ - r_zero_adj ;
      } else
      {
        // We're turning left
        start_pos = AZ_0;
        end_zone = L_correct[0];
        current_AZ = current_AZ - l_zero_adj ;
      }
    }
  }

  if (zone == 11)
  {
    if (moving)
    {
      // we're moving - use moving correction factors
      if (rotate_direction == "R")
      {
        start_zone = R_correct[zone - 1];
        end_zone = R_correct[zone];
        current_AZ = current_AZ  - r_max_adj;
      } else
      {
        // We're turning left - use left corrections
        end_pos = AZ_MAX;
        start_zone = L_correct[11];
        end_zone = l_max_adj;
        current_AZ = current_AZ  - l_max_adj;
      }
    } else
    {
      start_zone = AZ_Correction[zone - 1];
      end_zone = AZ_Correction[zone];
      calculated_AZ = current_AZ - AZ_0;
    }
  }


  calculated_AZ = current_AZ;

  if (zone > 0 && zone < 11)
    if (moving)
    {
      // Add the moving correction value when moving
      if (rotate_direction == "R")
      {
        start_zone = AZ_Correction[zone - 1];
        end_zone = R_correct[zone];
      } else
      {
        // We're turning left - use left corrections
        start_zone = AZ_Correction[zone - 1];
        end_zone =  L_correct[zone];
      }
    } else
    {
      start_zone = AZ_Correction[zone - 1];
      end_zone = AZ_Correction[zone];
      calculated_AZ = current_AZ - AZ_0;
    }

#ifdef debug1
  Serial.print("  Zone data: ");
  Serial.print(start_zone);
  Serial.print(" ");
  Serial.print(end_zone);
  Serial.print("  Zone range: ");
  Serial.print(start_pos);
  Serial.print(" ");
  Serial.print(end_pos);
#endif

  // Map the correction factor based on position in zone

  correction = map(calculated_AZ, start_pos, end_pos, start_zone, end_zone); // Map the correction

  current_AZ = current_AZ + correction;

  if (current_AZ < 0)
  {
    current_AZ = 0;
  }

#ifdef debug1
  Serial.print(" Adj: ");
  Serial.print(correction);
  Serial.print(" Final AZ: ");
  Serial.print(current_AZ);

#endif


}

void map_az()
{
  AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 360); // Map the current_AZ to calibrated degrees

  if (AZ_Degrees < 0) // Limit Zero point to 0
  {
    AZ_Degrees = 0;
  }

  if (AZ_Degrees > 360) // Limit max point to 360
  {
    AZ_Degrees = 360;
  }

  if (AZ_Degrees < 180)  // Adjust for North-centered controller
  {
    AZ_Degrees = 180 + AZ_Degrees;
  } else
  {
    AZ_Degrees = AZ_Degrees - 180;
  }

}

void fix_180()
{
  if (AZ_Degrees >= 180 && AZ_Degrees <= 360  && zone != 11) // Make sure we don't rotate left beyond 180 and we're not in zone 11
  {
    if ((AZ_Degrees + AZ_Distance < 180) && (AZ_Distance < 0))
    {
      AZ_Distance = 360 + AZ_Distance;
    }
  } else {
    // We're in zone 4 thru 12 - Make sure we don't rotate right beyond 180

#ifdef debug
    Serial.print(" In Zone 4 thru 12");
#endif

    if ((AZ_Degrees + AZ_Distance > 180) && (AZ_Distance > 0))
    {
      AZ_Distance = -360 + AZ_Distance;
    }
  }
}
